/*
 * Copyright (c) 2006-2009 by Panasonic Corporation
 *
 * All rights reserved.
 *
 */
#include <stdio.h>

#include <directfb.h>

#include <fusion/fusion.h>
#include <fusion/shmalloc.h>

#include <core/core.h>
#include <core/coredefs.h>
#include <core/coretypes.h>
#include <core/layers.h>
#include <core/screens.h>
#include <core/palette.h>
#include <core/surface.h>
#include <core/surface_buffer.h>
#include <core/system.h>

#include <core/state.h>
#include "surfacemanager.h"

#include <gfx/convert.h>

#include <misc/conf.h>
#include <misc/util.h>

#include <direct/debug.h>
#include <direct/memcpy.h>
#include <direct/messages.h>
#include <direct/thread.h>

#include "dfosd.h"
#include "primary.h"

#include <unistd.h>

D_DEBUG_DOMAIN( DFOSD_PRIMARY, "DFOSD/PRIMARY", "OSD Primary Surface" );

extern DFOSD_DEVICE  *dfosd_device;
extern DFOSD_DEVICE  dfosd_device_priv;
extern int dfosd_plane_reconfig(DFOSD_PLANE *plane, CoreLayerRegionConfig *config, CoreLayerRegionConfigFlags updated);


/*****************************************************************************/

static DFBResult
set_palette( DFOSD_PLANE *plane, CorePalette *palette );

static void
update_region( CoreLayer *layer, CoreSurface *surface, DFBRegion *regon, DFBSurfaceFlipFlags  flags );

/*****************************************************************************/

static DFBResult
primaryInitScreen( CoreScreen           *screen,
                   CoreGraphicsDevice   *device,
                   void                 *driver_data,
                   void                 *screen_data,
                   DFBScreenDescription *description )
{
  D_DEBUG_AT(DFOSD_PRIMARY, "%s \n", __FUNCTION__);

  /* Set the screen capabilities. */
  description->caps = DSCCAPS_NONE;

  /* Set the screen name. */
  snprintf( description->name,
            DFB_SCREEN_DESC_NAME_LENGTH, "DFOSD Primary Screen" );

  return DFB_OK;
}

#include <core/screens_internal.h>
static DFBResult
primaryGetScreenSize( CoreScreen *screen,
                      void       *driver_data,
                      void       *screen_data,
                      int        *ret_width,
                      int        *ret_height )
{
  D_DEBUG_AT(DFOSD_PRIMARY,"%s \n", __FUNCTION__);

  *ret_width  = dfosd_device->planes[screen->shared->screen_id].xres;
  *ret_height = dfosd_device->planes[screen->shared->screen_id].yres;

  return DFB_OK;
}

ScreenFuncs dfosdPrimaryScreenFuncs = {
  .InitScreen    = primaryInitScreen,
  .GetScreenSize = primaryGetScreenSize
};

/*****************************************************************************/

static int
primaryLayerDataSize()
{
  D_DEBUG_AT(DFOSD_PRIMARY,"%s \n", __FUNCTION__);

  return 0;
}

static int
primaryRegionDataSize()
{
  D_DEBUG_AT(DFOSD_PRIMARY,"%s \n", __FUNCTION__);

  return 0;
}

static DFBResult
primaryInitLayer( CoreLayer                  *layer,
                  void                       *driver_data,
                  void                       *layer_data,
                  DFBDisplayLayerDescription *description,
                  DFBDisplayLayerConfig      *config,
                  DFBColorAdjustment         *adjustment )
{
  DFOSD_PLANE *plane = (DFOSD_PLANE *)driver_data;

  D_DEBUG_AT(DFOSD_PRIMARY,"%s \n", __FUNCTION__);

  /* set capabilities and type */
  description->caps = DLCAPS_SURFACE;
  description->type = DLTF_GRAPHICS;

  /* set name */
  snprintf( description->name,
            DFB_DISPLAY_LAYER_DESC_NAME_LENGTH, "DFOSD Primary Layer" );

  /* fill out the default configuration */
  config->flags       = DLCONF_WIDTH       | DLCONF_HEIGHT |
    DLCONF_PIXELFORMAT | DLCONF_BUFFERMODE;
  config->buffermode  = DLBM_FRONTONLY;

  /* set layer size */
  config->width  = plane->xres;
  config->height = plane->yres;

  /* set pixelformat */
  config->pixelformat = plane->pixelformat;

  return DFB_OK;
}

static DFBResult
primaryTestRegion( CoreLayer                  *layer,
                   void                       *driver_data,
                   void                       *layer_data,
                   CoreLayerRegionConfig      *config,
                   CoreLayerRegionConfigFlags *failed )
{
  CoreLayerRegionConfigFlags fail = 0;

  D_DEBUG_AT(DFOSD_PRIMARY,"%s \n", __FUNCTION__);

  if (failed)
    *failed = fail;

  if((config->source.w < 1) || (config->source.h < 1) ||
     (config->source.w > 4096) || (config->source.h > 4096)){
	  D_DEBUG_AT(DFOSD_PRIMARY,"%s DFB_UNSUPPORTED\n", __FUNCTION__);
	  fail = 1;
  }

  if (fail)
    return DFB_UNSUPPORTED;
  return DFB_OK;
}

static DFBResult
primaryAddRegion( CoreLayer             *layer,
                  void                  *driver_data,
                  void                  *layer_data,
                  void                  *region_data,
                  CoreLayerRegionConfig *config )
{
  D_DEBUG_AT(DFOSD_PRIMARY,"%s \n", __FUNCTION__);

  return DFB_OK;
}

static DFBResult
primarySetRegion( CoreLayer                  *layer,
                  void                       *driver_data,
                  void                       *layer_data,
                  void                       *region_data,
                  CoreLayerRegionConfig      *config,
                  CoreLayerRegionConfigFlags  updated,
                  CoreSurface                *surface,
                  CorePalette                *palette,
                  CoreSurfaceBufferLock  *lock )
{
  DFBResult        ret = DFB_OK;
  DFOSD_PLANE     *plane = (DFOSD_PLANE *)driver_data;

  D_DEBUG_AT(DFOSD_PRIMARY,"%s \n", __FUNCTION__);

  if(dfosd_plane_reconfig(plane, config, updated)){
    return DFB_FAILURE;
  }

  if (surface) {
    plane->primary = surface;
  }

  if (palette && plane->primary) {
    ret = set_palette( plane, palette );
  }

  return ret;
}

static DFBResult
primaryRemoveRegion( CoreLayer             *layer,
                     void                  *driver_data,
                     void                  *layer_data,
                     void                  *region_data )
{
  D_DEBUG_AT(DFOSD_PRIMARY,"%s \n", __FUNCTION__);
  return DFB_OK;
}

static DFBResult
primaryFlipRegion( CoreLayer           *layer,
                   void                *driver_data,
                   void                *layer_data,
                   void                *region_data,
                   CoreSurface         *surface,
                   DFBSurfaceFlipFlags  flags,
                   CoreSurfaceBufferLock *lock )
{
  DFOSD_PLANE  *plane = (DFOSD_PLANE *)driver_data;

  D_DEBUG_AT(DFOSD_PRIMARY,"%s \n", __FUNCTION__);
  
  if(!plane->manager){
    return DFB_INIT;
  }
 
  /* update */
  fusion_skirmish_prevail(&plane->lock) ;

  update_region( layer, surface, NULL , flags);

  fusion_skirmish_dismiss(&plane->lock) ;

  return DFB_OK;
}

static DFBResult
primaryUpdateRegion( CoreLayer           *layer,
                     void                *driver_data,
                     void                *layer_data,
                     void                *region_data,
                     CoreSurface         *surface,
                     const DFBRegion     *update,
                     CoreSurfaceBufferLock *lock )
{
  DFOSD_PLANE  *plane = (DFOSD_PLANE *)driver_data;

  D_DEBUG_AT(DFOSD_PRIMARY,"%s \n", __FUNCTION__);

  if(!plane->manager){
    return DFB_INIT;
  }

  fusion_skirmish_prevail(&plane->lock) ;

  if (update) {
    DFBRegion region = *update;

    /*update*/
    update_region(layer, surface, &region, DSFLIP_NONE );
  }
  else {
    /*update*/
    update_region(layer, surface, NULL, DSFLIP_NONE );
  }

  fusion_skirmish_dismiss(&plane->lock) ;

  return DFB_OK;
}



DisplayLayerFuncs dfosdPrimaryLayerFuncs = {
  .LayerDataSize     = primaryLayerDataSize,
  .RegionDataSize    = primaryRegionDataSize,
  .InitLayer         = primaryInitLayer,
  .TestRegion        = primaryTestRegion,
  .AddRegion         = primaryAddRegion,
  .SetRegion         = primarySetRegion,
  .RemoveRegion      = primaryRemoveRegion,
  .FlipRegion        = primaryFlipRegion,
  .UpdateRegion      = primaryUpdateRegion,
};

/******************************************************************************/

static DFBResult
set_palette( DFOSD_PLANE *plane, CorePalette *palette )
{
  int            ret = DFB_OK;
  DFBColor      *colors = palette->entries;
  unsigned long  a, r, g, b, y, u, v;
  unsigned long *tmp_addr = (unsigned long *)plane->lut_pbase;
  int            i;

  if (!tmp_addr) {
    return DFB_OK;
  }
  plane->palette = palette;
  dfb_surface_set_palette( plane->primary, palette );

  for (i = 0; i < 256; i++) {
    a = colors[i].a;
    r = colors[i].r;
    g = colors[i].g;
    b = colors[i].b;

    y = 16829 * r + 33039 * g + 6416 * b;
    u = -4853 * r - 9530 * g + 14383 * b;
    v = 14386 * r - 12046 * g - 2340 * b;

    y = (y >> 16) + 16;
    u = (u >> 16) + 128;
    v = (v >> 16) + 128;

    *tmp_addr = ((a & 0xff)<<24) | ((y & 0xff)<<16)
      | ((u & 0xff)<<8) | (v & 0xff);
    tmp_addr++;
  }

  return ret;
}

static void
update_region(CoreLayer *layer, CoreSurface *surface,  DFBRegion *region, DFBSurfaceFlipFlags flag)
{
  DFBRectangle srect, drect;
  CoreSurfaceBufferRole from, to;
  CardState *state;

  state = dfb_layer_state(layer);

  if(!surface){
    return;
  }

  if(surface->num_buffers < 2){
    if(!(flag & DSFLIP_ONSYNC)){
     dfb_gfxcard_sync();
    return;
    }
  }

  from = state->from;
  to   = state->to;
  state->to=CSBR_FRONT;   
  state->from=CSBR_BACK;   

  state->clip.x1 = 0;
  state->clip.y1 = 0;
  state->clip.x2 = surface->config.size.w - 1;
  state->clip.y2 = surface->config.size.h - 1;

  if (region) {
    srect.x = region->x1;
    srect.y = region->y1;
    srect.w = region->x2 - region->x1 + 1;
    srect.h = region->y2 - region->y1 + 1;
    drect = srect;
  }
  else {
    drect.x = 0;
    drect.y = 0;
    drect.w = surface->config.size.w;
    drect.h = surface->config.size.h;
    srect = drect;
  }

  dfb_state_set_blitting_flags( state, DSBLIT_NOFX );
  dfb_state_set_source( state, surface); 
  dfb_state_set_destination( state, surface);
  dfb_gfxcard_blit( &srect, drect.x, drect.y, state );
  if(!(flag & DSFLIP_ONSYNC)){
  dfb_gfxcard_sync();
  }
  dfb_state_set_source( state, NULL); 
  dfb_state_set_destination( state, NULL);

  state->from=from;   
  state->to=to;   
}
