/*
 * Copyright (c) 2005, 2006 Sharp Corporation
 */

#include <config.h>

#include <stdio.h>

#include <directfb.h>

#include <core/core.h>
#include <core/coredefs.h>
#include <core/coretypes.h>
#include <core/layers.h>
#include <core/surfaces.h>
#include <core/system.h>

#include <gfx/convert.h>

#include <misc/conf.h>

#include <direct/mem.h>
#include <direct/memcpy.h>
#include <direct/messages.h>

#include <string.h>
#include <stdlib.h>

#include "shdfb.h"
#include "shdfb_system_primary.h"

extern dSHDFB_t  *ptSHDFB;
extern CoreDFB  *ptSHDFBCore;

/* local functions */

static DFBResult dfb_shdfb_set_video_mode (CoreSurface *ptSurface, CoreLayerRegionConfig *ptConfig );

/* Screen related functions stored into global data */

static DFBResult
primaryInitScreen (CoreScreen *ptScreen,
		                   GraphicsDevice *ptDevice,
    		               void *pvDriverData,
    		               void *pvScreenData,
		                   DFBScreenDescription *ptDescription)
{
     ptDescription->caps = DSCCAPS_NONE;

     snprintf (ptDescription->name, DFB_SCREEN_DESC_NAME_LENGTH, "SHDFB Primary Screen");

     return DFB_OK;
}

static DFBResult
primaryGetScreenSize (CoreScreen *ptScreen,
			                      void *pvDriverData,
			                      void *pvScreenData,
			                      int32       *pitRetWidth,
			                      int32       *piRetHeight)
{
     D_ASSERT (ptSHDFB != NULL);

     if (dfb_config->mode.width)
         *pitRetWidth  = dfb_config->mode.width;
     else
         *pitRetWidth  = ptSHDFB->uiWidth;;

     if (dfb_config->mode.height)
         *piRetHeight = dfb_config->mode.height;
     else
         *piRetHeight = ptSHDFB->uiHeight;

     return DFB_OK;
}

ScreenFuncs tShdfbPrimaryScreenFuncs = {
     .InitScreen    = primaryInitScreen,
     .GetScreenSize = primaryGetScreenSize
};

/* Layer related functions stored into global data */

static int32
primaryLayerDataSize()
{
     return 0;
}

static int32
primaryRegionDataSize()
{
     return 0;
}

static DFBResult
primaryInitLayer (CoreLayer *ptLayer,
		                  void *pvDriverData,
		                  void *pvLayerData,
		                  DFBDisplayLayerDescription *ptDescription,
		                  DFBDisplayLayerConfig *ptConfig,
		                  DFBColorAdjustment *ptAdjustment)
{
     ptDescription->caps = DLCAPS_SURFACE;
     ptDescription->type = DLTF_GRAPHICS;

     snprintf (ptDescription->name, DFB_DISPLAY_LAYER_DESC_NAME_LENGTH, "SHDFB Primary Layer");

     ptConfig->flags = DLCONF_WIDTH | DLCONF_HEIGHT | DLCONF_PIXELFORMAT | DLCONF_BUFFERMODE;
     ptConfig->buffermode  = DLBM_FRONTONLY;

     if (dfb_config->mode.width) {
          ptConfig->width  = dfb_config->mode.width;
     } else {
          ptConfig->width  = ptSHDFB->uiWidth;
     }

     if (dfb_config->mode.height) {
          ptConfig->height = dfb_config->mode.height;
     } else {
          ptConfig->height = ptSHDFB->uiHeight;
     }

     if (dfb_config->mode.format != DSPF_UNKNOWN) {
          ptConfig->pixelformat = dfb_config->mode.format;
     } else if (dfb_config->mode.depth > 0) {
          ptConfig->pixelformat = dfb_pixelformat_for_depth (dfb_config->mode.depth);
     } else { 
          ptConfig->pixelformat = SHDFB_FORMAT;
	  }

     return DFB_OK;
}

static DFBResult
primaryTestRegion (CoreLayer *ptLayer,
		                   void *pvDriverData,
		                   void *pvLayerData,
		                   CoreLayerRegionConfig *ptConfig,
		                   CoreLayerRegionConfigFlags *ptFailed)
{
     CoreLayerRegionConfigFlags tFail = 0;

     switch (ptConfig->buffermode) {
          case DLBM_FRONTONLY:
          case DLBM_BACKSYSTEM:
          case DLBM_BACKVIDEO:
               break;

          default:
               tFail |= CLRCF_BUFFERMODE;
               break;
     }

     if (ptConfig->options) {
          tFail |= CLRCF_OPTIONS;
     }

     if (ptFailed) {
          *ptFailed = tFail;
     }

     if (tFail) {
          return DFB_UNSUPPORTED;
     }

     return DFB_OK;
}

static DFBResult
primaryAddRegion (CoreLayer *ptLayer,
			                  void *pvDriverData,
			                  void *pvLayerData,
			                  void *pvRegionData,
			                  CoreLayerRegionConfig *ptConfig)
{
     return DFB_OK;
}

static DFBResult
primarySetRegion (CoreLayer *ptLayer,
			                  void *pvDriverData,
			                  void *pvLayerData,
			                  void *pvRegionData,
			                  CoreLayerRegionConfig *ptConfig,
			                  CoreLayerRegionConfigFlags  tUpdated,
			                  CoreSurface *ptSurface,
			                  CorePalette *ptPalette)
{
     DFBResult tRet;

     if (tUpdated & (CLRCF_BUFFERMODE | CLRCF_FORMAT | CLRCF_HEIGHT | CLRCF_SURFACE | CLRCF_WIDTH)) {
          tRet = dfb_shdfb_set_video_mode (ptSurface, ptConfig);
          if (tRet) {
               return tRet;
          }
     }

     return DFB_OK;
}

static DFBResult
primaryRemoveRegion (CoreLayer *ptLayer,
				                     void *pvDriverData,
				                     void *pvLayerData,
				                     void *pvRegionData)
{
		
     return DFB_OK;
}

static DFBResult
primaryFlipRegion (CoreLayer *ptLayer,
		                   void *pvDriverData,
		                   void *pvLayerData,
		                   void *pvRegionData,
		                   CoreSurface *ptSurface,
		                   DFBSurfaceFlipFlags tFlags)
{
	 D_DEBUG ("%s\n", __FUNCTION__);
     return DFB_OK;
}

static DFBResult
primaryUpdateRegion (CoreLayer *tLlayer,
			                     void *pvDriverData,
			                     void *pvLayerData,
			                     void *pvRegionData,
			                     CoreSurface *ptSurface,
			                     const DFBRegion *ptUpdate)
{
     return DFB_OK;
}

static DFBResult
primaryAllocateSurface (CoreLayer *ptLayer,
			                        void *pvDriverData,
			                        void *pvLayerData,
			                        void *pvRegionData,
			                        CoreLayerRegionConfig *ptConfig,
			                        CoreSurface **ptRet_ptSurface)
{
     DFBResult tRet;
     CoreSurface *ptSurface;
     DFBSurfaceCapabilities tCaps = DSCAPS_VIDEOONLY;
     
     D_DEBUG ("%s\n", __FUNCTION__);

     ptSurface = dfb_core_create_surface (ptSHDFBCore);
     if (!ptSurface) {
          return DFB_FAILURE;
     }

     ptSurface->idle_buffer = ptSurface->back_buffer = ptSurface->front_buffer = D_CALLOC (1, sizeof(SurfaceBuffer));

     if (!ptSurface->front_buffer) {
          fusion_object_destroy (&ptSurface->object);
          return DFB_NOSYSTEMMEMORY;
     }

     tRet = dfb_surface_init (ptSHDFBCore, ptSurface, ptConfig->width, ptConfig->height, ptConfig->format, tCaps, NULL);
     if (tRet) {
          D_FREE (ptSurface->front_buffer);
          fusion_object_destroy (&ptSurface->object);
          return tRet;
     }

     fusion_object_activate (&ptSurface->object);

     *ptRet_ptSurface = ptSurface;

     return DFB_OK;
}

static DFBResult
primaryReallocateSurface (CoreLayer *ptLayer,
				                          void *pvDriverData,
				                          void *pvLayerData,
				                          void *pvRegionData,
				                          CoreLayerRegionConfig *ptConfig,
				                          CoreSurface *ptSurface)
{
     return DFB_OK;
}

DisplayLayerFuncs tShdfbPrimaryLayerFuncs = {
     .LayerDataSize     = primaryLayerDataSize,
     .RegionDataSize    = primaryRegionDataSize,
     .InitLayer         = primaryInitLayer,

     .TestRegion        = primaryTestRegion,
     .AddRegion         = primaryAddRegion,
     .SetRegion         = primarySetRegion,
     .RemoveRegion      = primaryRemoveRegion,
     .FlipRegion        = primaryFlipRegion,
     .UpdateRegion      = primaryUpdateRegion,

     .AllocateSurface   = primaryAllocateSurface,
     .ReallocateSurface = primaryReallocateSurface,
};

/* local functions */

static DFBResult
dfb_shdfb_set_video_mode (CoreSurface *ptSurface, CoreLayerRegionConfig *ptConfig )
{
     D_DEBUG("dfb_shdfb_set_video_mode\n");

     D_ASSERT (ptConfig != NULL);

     if (ptSurface) {
          ptSurface->width  = ptConfig->width;
          ptSurface->height = ptConfig->height;
          ptSurface->format = ptConfig->format;

          ptSurface->front_buffer->surface = ptSurface;
          ptSurface->front_buffer->policy = CSP_VIDEOONLY;
          ptSurface->front_buffer->format = ptConfig->format;
          ptSurface->front_buffer->system.health = CSH_INVALID;
          ptSurface->front_buffer->video.health = CSH_STORED;
          ptSurface->front_buffer->video.pitch = ((dLnxDFBAddr_t *)ptSurface->front_buffer->system.addr)->uiPitch;

          switch (ptConfig->buffermode) {
               case DLBM_FRONTONLY:
               case DLBM_BACKSYSTEM:
               case DLBM_BACKVIDEO:
                    break;
               default:
                    D_BUG ("unexpected buffer mode");
                    break;
          }

          ptSHDFB->ptAddr[ptSurface->front_buffer->video.offset] = (void *) ptSurface->front_buffer->system.addr;

          dfb_surface_notify_listeners (ptSurface, CSNF_SIZEFORMAT | CSNF_FLIP | CSNF_VIDEO | CSNF_SYSTEM);
     }

	return DFB_OK;
}
