/*
 * Copyright (c) 2005, 2006 Sharp Corporation
 */

#include <config.h>

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <directfb.h>

#include <core/fonts.h>
#include <core/gfxcard.h>
#include <core/surfaces.h>
#include <core/surfacemanager.h>

#include <gfx/convert.h>

#include <media/idirectfbfont.h>

#include <direct/mem.h>
#include <direct/memcpy.h>
#include <direct/messages.h>
#include <direct/util.h>

#include <misc/conf.h>
#include <misc/util.h>

#include "lnxdfbc.h"


static DFBResult
Probe( IDirectFBFont_ProbeContext *ctx );

static DFBResult
Construct( IDirectFBFont      *thiz,
           CoreDFB            *core,
           const char         *filename,
           DFBFontDescription *desc );

#include <direct/interface_implementation.h>

DIRECT_INTERFACE_IMPLEMENTATION( IDirectFBFont, SHDFB )

typedef struct {
	dLnxDFBFont_t face;
	int fixed_advance;
} SHDFBImplData;

static DFBResult
render_glyph( CoreFont      *thiz,
              unichar        glyph,
              CoreGlyphData *info,
              CoreSurface   *surface )
{
     __u8        *src;
     __u8        *dst;
     int          y;
     int          pitch;
     dLnxDFBFont_t *face;
     SHDFBImplData *data = (SHDFBImplData*) thiz->impl_data;
     DFBResult err;
     
     D_DEBUG( "DirectFB/FontSHDFB: "
      					"%s\n", __FUNCTION__);

	 face = &data->face;
     if (iLnxDFBGetFontGlyph ( face, glyph ) < 0) {
          D_DEBUG( "DirectFB/FontSHDFB: "
                         "Could not render glyph for character #%d!\n", glyph );
          return DFB_FAILURE;
     }
     
     D_DEBUG ("%s: get glyph\n", __FUNCTION__);

     err = dfb_surface_soft_lock( surface, DSLF_WRITE,
                                  &dst, &pitch, 0 );
     if (err) {
          D_ERROR( "DirectB/FontSHDFB: Unable to lock surface!\n" );
          return err;
     }

     info->width = face->width;
     if (info->width + thiz->next_x > surface->width)
          info->width = surface->width - thiz->next_x;

     info->height = face->height;
     if (info->height > surface->height)
          info->height = surface->height;

     info->left = face->left;
     info->top  = thiz->ascender - face->top;
     
     D_DEBUG( "DirectFB/FontSHDFB: "
      					"%s: top = %d, %d\n", __FUNCTION__, info->top, thiz->next_x);

	 src = face->bitmap;
     dst += DFB_BYTES_PER_LINE(surface->format, thiz->next_x);

     for (y=0; y < info->height; y++) {
          int    i, j, n;
          __u8  *dst8  = dst;
          __u16 *dst16 = (__u16 *) dst;
          __u32 *dst32 = (__u32 *) dst;

          switch (surface->format) {
#if 0  /* TODO: not implemented */
                 case DSPF_ARGB:
                      if (thiz->surface_caps & DSCAPS_PREMULTIPLIED) {
                           for (i=0; i<info->width; i++)
                               dst32[i] = ((src[i] << 24) |
                                            (src[i] << 16) |
                                            (src[i] <<  8) | src[i]);
                      }
                      else
                           for (i=0; i<info->width; i++)
                                dst32[i] = (src[i] << 24) | 0xFFFFFF;
                      break;
#endif                                
                 case DSPF_A8:
                       for (i=0; i<info->width; i++) {
                       	    int val  = src[i / 4];
                       	    if (val) {
	                       		int p = 6 - 2 * (i % 4);
	                       		val =( val & (0x3 << p)) >> p;
	                            dst8[i]  = val * 85; /* 85 = 255 / 3*/
                       	    } else {
                       	    	dst8[i] = 0;
                       	    }
                       }
                      break;
                 default:
                      break;
            }

          src += face->pitch;
          dst += pitch;
     }
     
     dfb_surface_unlock( surface, 0 );

     return DFB_OK;
}


static DFBResult
get_glyph_info( CoreFont      *thiz,
                unichar        glyph,
                CoreGlyphData *info )
{
     SHDFBImplData *data = (SHDFBImplData*) thiz->impl_data;
     dLnxDFBFont_t *face;

     D_DEBUG( "DirectFB/FontSHDFB: "
      					"%s\n", __FUNCTION__);

	 face = &data->face;
     if (iLnxDFBGetFontInfo ( face, glyph ) < 0) {
          D_DEBUG( "DirectB/FontSHDFB: "
                         "Could not load glyph for character #%d!\n", glyph );

          return DFB_FAILURE;
     }

     info->width   = face->width;
     info->height  = face->height;
     info->advance = data->fixed_advance ?
                     data->fixed_advance : face->advance_x;

     D_DEBUG( "DirectFB/FontSHDFB: "
      					"%s: %d %d %d\n", __FUNCTION__, info->width, info->height, info->advance);
     return DFB_OK;
}

#if 0
static DFBResult
get_kerning( CoreFont *thiz,
             unichar   prev,
             unichar   current,
             int      *kern_x,
             int      *kern_y)
{
     int x, y;
     SHDFBImplData *data = (SHDFBImplData*) thiz->impl_data;
     dLnxDFBFont_t *face;

     D_ASSUME( (kern_x != NULL) || (kern_y != NULL) );

	 face = &data->face;

     /* Lookup kerning values for the character pair. */
     iLnxDFBGetKerning (face, prev, current, &x, &y);

     if (kern_x)
          *kern_x = x;

     if (kern_y)
          *kern_y = y;

     return DFB_OK;
}
#endif

static void
IDirectFBFont_SHDFB_Destruct( IDirectFBFont *thiz )
{
     IDirectFBFont_data *data = (IDirectFBFont_data*)thiz->priv;

     if (data->font->impl_data) {
          SHDFBImplData *impl_data = (SHDFBImplData*) data->font->impl_data;

          D_FREE( impl_data );

          data->font->impl_data = NULL;
     }

     IDirectFBFont_Destruct( thiz );
}


static DFBResult
IDirectFBFont_SHDFB_Release( IDirectFBFont *thiz )
{
     DIRECT_INTERFACE_GET_DATA(IDirectFBFont)

     if (--data->ref == 0) {
          IDirectFBFont_SHDFB_Destruct( thiz );
     }

     return DFB_OK;
}


static DFBResult
Probe( IDirectFBFont_ProbeContext *ctx )
{
     D_DEBUG( "DirectFB/FontSHDFB: Probe font `%s'.\n", ctx->filename );

     return DFB_OK;
}


static DFBResult
Construct( IDirectFBFont      *thiz,
           CoreDFB            *core,
           const char         *filename,
           DFBFontDescription *desc )
{
     CoreFont              *font;
     dLnxDFBFont_t            *face;
     SHDFBImplData           *data;
 
     D_DEBUG( "DirectFB/FontSHDFB: "
                    "Construct font from file `%s' (index %d) at pixel size %d x %d.\n",
                    filename,
                    (desc->flags & DFDESC_INDEX)  ? desc->index  : 0,
                    (desc->flags & DFDESC_WIDTH)  ? desc->width  : 0,
                    (desc->flags & DFDESC_HEIGHT) ? desc->height : 0 );

     data = D_CALLOC( 1, sizeof(SHDFBImplData) );
     face = &data->face;
     
     if (desc->flags & (DFDESC_HEIGHT       | DFDESC_WIDTH |
                        DFDESC_FRACT_HEIGHT | DFDESC_FRACT_WIDTH))
     {
          int fw = 0, fh = 0;

          if (desc->flags & DFDESC_FRACT_HEIGHT)
               fh = desc->fract_height;
          else if (desc->flags & DFDESC_HEIGHT)
               fh = desc->height;

          if (desc->flags & DFDESC_FRACT_WIDTH)
               fw = desc->fract_width;
          else if (desc->flags & DFDESC_WIDTH)
               fw = desc->width;

          if (iLnxDFBSetFontSize ( face, fw, fh ) < 0) {
               D_ERROR( "DirectB/FontSHDFB: "
                         "Could not set pixel size to %d x %d!\n",
                         (desc->flags & DFDESC_WIDTH)  ? desc->width  : 0,
                         (desc->flags & DFDESC_HEIGHT) ? desc->height : 0 );
               D_FREE (data);
               DIRECT_DEALLOCATE_INTERFACE( thiz );
               return DFB_FAILURE;
          }
     }

     font = dfb_font_create( core );

#if 0
     D_ASSERT( font->pixel_format == DSPF_ARGB ||
               font->pixel_format == DSPF_A8);
#else
     D_ASSERT( font->pixel_format == DSPF_A8);
#endif               

     font->ascender   = face->ascender;
     font->descender  = face->descender;
     font->height     = font->ascender + ABS(font->descender) + 1;
     font->maxadvance = face->max_advance;

     D_DEBUG( "DirectFB/FontSHDFB: font->height = %d\n", font->height );
     D_DEBUG( "DirectFB/FontSHDFB: font->ascender = %d\n", font->ascender );
     D_DEBUG( "DirectFB/FontSHDFB: font->descender = %d\n",font->descender );

     font->GetGlyphInfo = get_glyph_info;
     font->RenderGlyph  = render_glyph;

#if 0
     font->GetKerning = get_kerning;
#endif     

     if (desc->flags & DFDESC_FIXEDADVANCE) {
          data->fixed_advance = desc->fixed_advance;
          font->maxadvance    = desc->fixed_advance;
     }

     font->impl_data = data;

     IDirectFBFont_Construct( thiz, font );

     thiz->Release = IDirectFBFont_SHDFB_Release;

     return DFB_OK;
}
