/*
 * Copyright (c) 2005, 2006 Sharp Corporation
 */

#include <config.h>

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#include <directfb.h>

#include <core/layers.h>

#include <misc/conf.h>

#include "shdfb_gfx.h"

DisplayLayerFuncs  tShdfbOldPrimaryFuncs;
void *pShdfbOldPrimaryDriverData;

static DFBResult
shdfbInitLayer (CoreLayer *ptLayer,
              void *pDriver_data,
              void *pLayer_data,
              DFBDisplayLayerDescription *ptDescription,
              DFBDisplayLayerConfig *ptConfig,
              DFBColorAdjustment *ptAdjustment)
{
    DFBResult tRet;
    dShdfbDriverData_t *ptDriver = (dShdfbDriverData_t *) pDriver_data;
    int32 iRet;
	
    tRet = tShdfbOldPrimaryFuncs.InitLayer (ptLayer, pShdfbOldPrimaryDriverData, pLayer_data, ptDescription, ptConfig, ptAdjustment);
	if (tRet) {
		return tRet;
	}

    snprintf (ptDescription->name, DFB_DISPLAY_LAYER_DESC_NAME_LENGTH, "SHDFB Graphics");

    ptDescription->caps |= DLCAPS_ALPHACHANNEL;
    ptConfig->pixelformat = DSPF_ARGB;

#ifndef EMULATION                          
    iRet = iLnxDFBInitPrimary (&ptDriver->tPrimaryAddr);                      
#else
	ptDriver->tPrimaryAddr.pdvAddr = NULL;
	ptDriver->reaction_num = 0;
	iRet = 0;
#endif

    ptConfig->width = ptDriver->tPrimaryAddr.uiWidth;
    ptConfig->height = ptDriver->tPrimaryAddr.uiHeight;

    return iRet == 0 ? DFB_OK : DFB_FAILURE;
}

static DFBResult
shdfbTestRegion (CoreLayer *ptLayer,
               void *pDriver_data,
               void *pLayer_data,
               CoreLayerRegionConfig *ptConfig,
               CoreLayerRegionConfigFlags *ptFailed)
{
    DFBResult tRet;
    CoreLayerRegionConfigFlags tFail = 0;
    DFBDisplayLayerOptions tOptions = ptConfig->options;

    ptConfig->options = DLOP_NONE;

    tRet = tShdfbOldPrimaryFuncs.TestRegion (ptLayer, pShdfbOldPrimaryDriverData, pLayer_data, ptConfig, &tFail);

    if (tOptions) {
		tFail |= CLRCF_OPTIONS;
    }

    ptConfig->options = tOptions;

    if (ptFailed) {
    	*ptFailed = tFail;
    }

    if (tFail) {
    	return DFB_UNSUPPORTED;
    }
	
    return tRet;
}

DFBResult
shdfbSetRegion (CoreLayer *ptLayer,
              void *pDriver_data,
              void *pLayer_data,
              void *pRegion_data,
              CoreLayerRegionConfig *ptConfig,
              CoreLayerRegionConfigFlags tUpdated,
              CoreSurface *ptSurface,
              CorePalette *ptPalette)
{
    dShdfbDriverData_t *ptDriver = (dShdfbDriverData_t *) pDriver_data;
	DFBResult tRet;

    tRet = tShdfbOldPrimaryFuncs.SetRegion (ptLayer, pShdfbOldPrimaryDriverData, 
                                        pLayer_data, pRegion_data,
                                        ptConfig, tUpdated, ptSurface,
                                        ptPalette);
    if (tRet) {
    	return tRet;
    }
    
    return DFB_OK;
}

#ifdef EMULATION
static DFBResult
shdfbPrimarySetRegion (CoreLayer *ptLayer,
              void *pDriver_data,
              void *pLayer_data,
              void *pRegion_data,
              CoreLayerRegionConfig *ptConfig,
              CoreLayerRegionConfigFlags tUpdated,
              CoreSurface *ptSurface,
              CorePalette *ptPalette)
{
    dShdfbDriverData_t *ptDriver = (dShdfbDriverData_t *) pDriver_data;
	DFBResult tRet;

    tRet = tShdfbOldPrimaryFuncs.SetRegion (ptLayer, pShdfbOldPrimaryDriverData, 
                                        pLayer_data, pRegion_data,
                                        ptConfig, tUpdated, ptSurface,
                                        ptPalette);
    if (tRet) {
    	return tRet;
    }
    
#ifdef EMULATION
#undef malloc
	if (ptDriver->tPrimaryAddr.pdvAddr == NULL) {
		ptDriver->tPrimaryAddr.pdvAddr = ptDriver->tRotationAddr.pdvAddr = malloc (1920 * 1080 * 4);
		ptDriver->tYUV_LAddr.pdvAddr = ptDriver->tYUV_PAddr.pdvAddr = malloc (1920 * 1080 * 2);
		ptDriver->tARGB_LAddr.pdvAddr = ptDriver->tARGB_PAddr.pdvAddr = malloc (1920 * 1080 * 4);
		ptDriver->tMainDisplayAddr.pdvAddr = ptDriver->tPrimaryAddr.pdvAddr;
		ptDriver->coreOfPrimarySurface = ptSurface;
    	fusion_reactor_attach (ptSurface->object.reactor, shdfbDestroySurface, ptDriver, &ptDriver->reaction[ptDriver->reaction_num++]);
	}
    ptSurface->front_buffer->system.addr = ptDriver->tPrimaryAddr.pdvAddr;
#endif

    return DFB_OK;
}
#endif

DFBResult 
shdfbRemoveRegion (CoreLayer *ptLayer,
                void *pDriver_data,
                void *pLayer_data,
                void *pRegion_data)
{
    return DFB_OK;
}

DFBResult
shdfbReallocateSurface (CoreLayer *ptLayer,
			                        void *pvDriverData,
			                        void *pvLayerData,
			                        void *pvRegionData,
			                        CoreLayerRegionConfig *ptConfig,
			                        CoreSurface *ptSurface)
{
    DFBResult tRet;
    dShdfbDriverData_t *ptDriver = (dShdfbDriverData_t *) pvDriverData;
    
    tRet =  tShdfbOldPrimaryFuncs.ReallocateSurface (ptLayer, pShdfbOldPrimaryDriverData, pvLayerData, pvRegionData, ptConfig, ptSurface);
    if (tRet) {
    	return tRet;
    }
    
    return DFB_OK;
}

static DFBResult
shdfbUpdateRegion (CoreLayer *ptLayer,
			                     void *pvDriverData,
			                     void *pvLayerData,
			                     void *pvRegionData,
			                     CoreSurface *ptSurface,
			                     const DFBRegion *ptUpdate)
{
#ifndef 	EMULATION
	int32 iRet;

	if (ptUpdate) {
		iRet = iLnxDFBFlip (ptUpdate->x1, ptUpdate->y1, 
								ptUpdate->x2 - ptUpdate->x1 + 1,  ptUpdate->y2 - ptUpdate->y1 + 1,
								ptSurface->front_buffer->system.addr);
	} else {
		iRet = iLnxDFBFlip (0, 0, ptSurface->width, ptSurface->height, ptSurface->front_buffer->system.addr);
	}

    return iRet == 0 ? DFB_OK : DFB_FAILURE;
#else    
    dShdfbDriverData_t *ptDriver = (dShdfbDriverData_t *) pvDriverData;
	if (ptUpdate) {
	    int y;
	    uint8 *dst8 = (uint8 *) ptDriver->tMainDisplayAddr.pdvAddr;
	    uint8 *src8 = (uint8 *) ptDriver->tPrimaryAddr.pdvAddr;
	    int width = ptUpdate->x2 - ptUpdate->x1 + 1;
	    
	    src8 += ptUpdate->y1 * ptSurface->width * 4 + ptUpdate->x1;
	    dst8 += ptUpdate->y1 * ptSurface->width * 4 + ptUpdate->x1;
	    for (y = ptUpdate->y1; y < ptUpdate->y2 + 1; y++) {
	    	memcpy (dst8, src8, width * 4);
	    	dst8 += ptSurface->width * 4;
	    	src8 += ptSurface->width * 4;
	    }
	} else {
		memcpy (ptDriver->tMainDisplayAddr.pdvAddr, ptDriver->tPrimaryAddr.pdvAddr, ptSurface->width * ptSurface->height * 4);
	}

    return  tShdfbOldPrimaryFuncs.UpdateRegion (ptLayer, pShdfbOldPrimaryDriverData, pvLayerData, pvRegionData, ptSurface, ptUpdate);
#endif    
}

static DFBResult
shdfbFlipRegion (CoreLayer *ptLayer,
			                     void *pvDriverData,
			                     void *pvLayerData,
			                     void *pvRegionData,
			                     CoreSurface *ptSurface,
				                 DFBSurfaceFlipFlags tFlags)
{
#ifndef 	EMULATION
	int32 iRet;

	iRet = iLnxDFBFlip (0, 0, ptSurface->width, ptSurface->height, ptSurface->front_buffer->system.addr);
    return iRet == 0 ? DFB_OK : DFB_FAILURE;
#else	
    dShdfbDriverData_t *ptDriver = (dShdfbDriverData_t *) pvDriverData;

	memcpy (ptDriver->tMainDisplayAddr.pdvAddr, ptDriver->tPrimaryAddr.pdvAddr, ptSurface->width * ptSurface->height * 4);

    return tShdfbOldPrimaryFuncs.FlipRegion (ptLayer, pShdfbOldPrimaryDriverData, pvLayerData, pvRegionData, ptSurface, tFlags);
#endif    
}

#ifdef EMULATION
ReactionResult 
shdfbDestroySurface ( const void *msg_data, void *ctx )
{
     const CoreSurfaceNotification *notification = msg_data;
     
     if (notification->flags == CSNF_DESTROY) {
     	dShdfbDriverData_t   *ptDriver = (dShdfbDriverData_t  *) ctx;
		CoreSurface *surface = ptDriver->coreOfPrimarySurface;
     	surface->front_buffer->system.health = CSH_INVALID;
#undef free     	
 		free (ptDriver->tPrimaryAddr.pdvAddr);
 		free (ptDriver->tMainDisplayAddr.pdvAddr);
 		free (ptDriver->tYUVAddr.pdvAddr);
 		free (ptDriver->tARGBAddr.pdvAddr);
     	return RS_REMOVE;
     }
     
     return RS_OK;
}

ReactionResult 
shdfbStopSurface ( const void *msg_data, void *ctx )
{
     const CoreSurfaceNotification *notification = msg_data;
     
     if (notification->flags == CSNF_DESTROY) {
     	CoreSurface *surface = (CoreSurface *) ctx;
     	surface->front_buffer->system.health = CSH_INVALID;
     	return RS_REMOVE;
     }
     
     return RS_OK;
}
#endif

static DFBResult
shdfbAllocateSurface (CoreLayer *ptLayer,
			                        void *pvDriverData,
			                        void *pvLayerData,
			                        void *pvRegionData,
			                        CoreLayerRegionConfig *ptConfig,
			                        CoreSurface **ptRetSurface)
{
    DFBResult tRet;
    dShdfbDriverData_t *ptDriver = (dShdfbDriverData_t *) pvDriverData;
    CoreSurface *ptSurface;
    
    tRet =  tShdfbOldPrimaryFuncs.AllocateSurface (ptLayer, pShdfbOldPrimaryDriverData, pvLayerData, pvRegionData, ptConfig, ptRetSurface);
    if (tRet) {
    	return tRet;
    }
    
    ptSurface = *ptRetSurface;

#ifndef EMULATION
    ptSurface->front_buffer->system.addr = &ptDriver->tPrimaryAddr;
	ptSurface->front_buffer->video.offset = 0;
#endif	
    
    return DFB_OK;
}

DisplayLayerFuncs tShdfbPrimaryFuncs = {
     InitLayer:		shdfbInitLayer,
     TestRegion:   shdfbTestRegion,
#ifndef EMULATION     
     SetRegion:    shdfbSetRegion,
#else     
     SetRegion:    shdfbPrimarySetRegion,
#endif     
	 UpdateRegion:    shdfbUpdateRegion,
	 FlipRegion:    shdfbFlipRegion,
     AllocateSurface: shdfbAllocateSurface,
};

