/*
 * Copyright (C) 2006, 2007, 2008, 2009, 2010 Apple Inc. All rights reserved.
 * Copyright (C) 2008 Collabora Ltd. All rights reserved.
 * Copyright (C) 2009 Girish Ramakrishnan <girish@forwardbias.in>
 * Copyright (C) 2009, 2010, 2011 Espial Group Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#ifndef PluginView_h
#define PluginView_h

#include "FrameLoadRequest.h"
#include "HaltablePlugin.h"
#include "IntRect.h"
#include "MediaCanStartListener.h"
#include "PluginStream.h"
#include "ResourceRequest.h"
#include "Timer.h"
#include "Widget.h"
#include "npruntime_internal.h"
#include <wtf/HashMap.h>
#include <wtf/HashSet.h>
#include <wtf/OwnPtr.h>
#include <wtf/PassRefPtr.h>
#include <wtf/RefPtr.h>
#include <wtf/Vector.h>
#include <wtf/text/CString.h>
#include <cairo.h>

#if OS(WINDOWS) && (PLATFORM(QT) || PLATFORM(WX))
typedef struct HWND__* HWND;
typedef HWND PlatformPluginWidget;
#else
typedef PlatformWidget PlatformPluginWidget;
#if defined(XP_MACOSX) && PLATFORM(QT)
#include <QPixmap>
#endif
#endif

namespace JSC {
    namespace Bindings {
        class Instance;
    }
}

namespace WebCore {
    class Element;
    class Frame;
    class Image;
    class KeyboardEvent;
    class MouseEvent;
    class KURL;
#if OS(WINDOWS) && ENABLE(NETSCAPE_PLUGIN_API)
    class PluginMessageThrottlerWin;
#endif
    class PluginPackage;
    class PluginRequest;
    class PluginStream;
    class ResourceError;
    class ResourceResponse;

#ifdef TPI_FLASH
    // NPN Timers for Flash
    class PluginTimers;
    typedef void (*TIMERFUNC)(NPP npp, uint32_t timerID);

    class PluginTimer : public TimerBase {
    public:
        PluginTimer(PluginTimer** timerList, NPP instance, bool repeat, TIMERFUNC timerFunc);
        virtual ~PluginTimer();

        uint32_t timerID() const { return m_timerID; }

        static PluginTimer* Find(PluginTimer* list, uint32_t timerID);

        PluginTimer* next() const { return m_next; }

    private:
        // override from TimerBase
        virtual void fired();

        friend class PluginTimerList;

        PluginTimer**   m_list;
        PluginTimer*    m_prev;
        PluginTimer*    m_next;
        NPP             m_instance;
        TIMERFUNC       m_timerFunc;
        uint32_t        m_timerID;
        bool            m_repeat;
    };

    class PluginTimers {
    public:
        PluginTimers() : m_TimerList(0) {}
        ~PluginTimers();

        uint32_t schedule(NPP instance, uint32_t interval, bool repeat, TIMERFUNC proc);
        void unschedule(NPP instance, uint32_t timerID);

    private:
        PluginTimer* m_TimerList;
    };
    // End NPN Timers for Flash

#endif

    enum PluginStatus {
        PluginStatusCanNotFindPlugin,
        PluginStatusCanNotLoadPlugin,
        PluginStatusLoadedSuccessfully
    };

    class PluginRequest : public Noncopyable {
    public:
        PluginRequest(const FrameLoadRequest& frameLoadRequest, bool sendNotification, void* notifyData, bool shouldAllowPopups)
            : m_frameLoadRequest(frameLoadRequest)
            , m_notifyData(notifyData)
            , m_sendNotification(sendNotification)
            , m_shouldAllowPopups(shouldAllowPopups) { }
    public:
        const FrameLoadRequest& frameLoadRequest() const { return m_frameLoadRequest; }
        void* notifyData() const { return m_notifyData; }
        bool sendNotification() const { return m_sendNotification; }
        bool shouldAllowPopups() const { return m_shouldAllowPopups; }
    private:
        FrameLoadRequest m_frameLoadRequest;
        void* m_notifyData;
        bool m_sendNotification;
        bool m_shouldAllowPopups;
    };

    class PluginManualLoader {
    public:
        virtual ~PluginManualLoader() {}
        virtual void didReceiveResponse(const ResourceResponse&) = 0;
        virtual void didReceiveData(const char*, int) = 0;
        virtual void didFinishLoading() = 0;
        virtual void didFail(const ResourceError&) = 0;
    };

    class PluginView : public Widget, private PluginStreamClient, public PluginManualLoader, private HaltablePlugin, private MediaCanStartListener {
    	friend class PluginViewAACAudio;
    public:
        static PassRefPtr<PluginView> create(Frame* parentFrame, const IntSize&, Element*, const KURL&, const Vector<String>& paramNames, const Vector<String>& paramValues, const String& mimeType, bool loadManually);
        virtual ~PluginView();

        PluginPackage* plugin() const { return m_plugin.get(); }
        NPP instance() const { return m_instance; }

        void setNPWindowRect(const IntRect&);
        static PluginView* currentPluginView();

        PassRefPtr<JSC::Bindings::Instance> bindingInstance();

        PluginStatus status() const { return m_status; }

        // NPN functions
        NPError getURLNotify(const char* url, const char* target, void* notifyData);
        NPError getURL(const char* url, const char* target);
        NPError postURLNotify(const char* url, const char* target, uint32_t len, const char* but, NPBool file, void* notifyData);
        NPError postURL(const char* url, const char* target, uint32_t len, const char* but, NPBool file);
        NPError newStream(NPMIMEType type, const char* target, NPStream** stream);
        int32_t write(NPStream* stream, int32_t len, void* buffer);
        NPError destroyStream(NPStream* stream, NPReason reason);
        const char* userAgent();
#if ENABLE(NETSCAPE_PLUGIN_API)
        static const char* userAgentStatic();
#endif
        void status(const char* message);
        
#if ENABLE(NETSCAPE_PLUGIN_API)
        NPError getValue(NPNVariable variable, void* value);
        static NPError getValueStatic(NPNVariable variable, void* value);
#endif
        NPError setValue(NPPVariable variable, void* value);
        void invalidateRect(NPRect*);
        void invalidateRegion(NPRegion);
        void forceRedraw();
        void pushPopupsEnabledState(bool state);
        void popPopupsEnabledState();

        virtual void invalidateRect(const IntRect&);

        bool arePopupsAllowed() const;

        void setJavaScriptPaused(bool);

        void privateBrowsingStateChanged(bool);

        void disconnectStream(PluginStream*);
        void streamDidFinishLoading(PluginStream* stream) { disconnectStream(stream); }

        // Widget functions
        virtual void setFrameRect(const IntRect&);
        virtual void frameRectsChanged();
        virtual void setFocus(bool);
        virtual void show();
        virtual void hide();
        virtual void paint(GraphicsContext*, const IntRect&);

        // This method is used by plugins on all platforms to obtain a clip rect that includes clips set by WebCore,
        // e.g., in overflow:auto sections.  The clip rects coordinates are in the containing window's coordinate space.
        // This clip includes any clips that the widget itself sets up for its children.
        IntRect windowClipRect() const;

        virtual void handleEvent(Event*);
        virtual void setParent(ScrollView*);
        virtual void setParentVisible(bool);

        virtual bool isPluginView() const { return true; }

        Frame* parentFrame() const { return m_parentFrame.get(); }

        void focusPluginElement();

        const String& pluginsPage() const { return m_pluginsPage; }
        const String& mimeType() const { return m_mimeType; }
        const KURL& url() const { return m_url; }

#if OS(WINDOWS) && ENABLE(NETSCAPE_PLUGIN_API)
        static LRESULT CALLBACK PluginViewWndProc(HWND, UINT, WPARAM, LPARAM);
        LRESULT wndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
        WNDPROC pluginWndProc() const { return m_pluginWndProc; }
#endif

        // Used for manual loading
        void didReceiveResponse(const ResourceResponse&);
        void didReceiveData(const char*, int);
        void didFinishLoading();
        void didFail(const ResourceError&);

        // HaltablePlugin
        virtual void halt();
        virtual void restart();
        virtual Node* node() const;
        virtual bool isWindowed() const { return m_isWindowed; }
        virtual String pluginName() const;

        bool isHalted() const { return m_isHalted; }
        bool hasBeenHalted() const { return m_hasBeenHalted; }

#ifndef TPI_FLASH
		 Vector<unsigned char> m_offScreenBuf;
#endif

        static bool isCallingPlugin();

		IntRect m_cairoSurfaceSize;
		cairo_surface_t* m_cairoSurface;
			
#ifdef TPI_FLASH
		bool m_fullscreen;
		bool m_skipOn2;
		bool m_skipSorenson;
		RefPtr<Image>  m_nullPluginImage;
		RefPtr<Image>  m_on2CodecImage;
		RefPtr<Image>  m_sorensonCodecImage;
		uint32_t m_codecType;
		void paintCodecTypeIcon(GraphicsContext* context, const IntRect& rect);
        // NPN Timers for Flash
        uint32_t  scheduleTimer(NPP instance, uint32_t interval, bool repeat, void (*timerFunc)(NPP instance, uint32_t timerID));
        void    unscheduleTimer(NPP instance, uint32_t timerID);
#endif
        
        bool start();
        void stop();

#if ENABLE(NETSCAPE_PLUGIN_API)
        static void keepAlive(NPP);
#endif
        void keepAlive();

    private:
        PluginView(Frame* parentFrame, const IntSize&, PluginPackage*, Element*, const KURL&, const Vector<String>& paramNames, const Vector<String>& paramValues, const String& mimeType, bool loadManually);

        void setParameters(const Vector<String>& paramNames, const Vector<String>& paramValues);
        bool startOrAddToUnstartedList();
        void removeFromUnstartedListIfNecessary();
        void init();
        bool platformStart();
        void platformDestroy();
        static void setCurrentPluginView(PluginView*);
        NPError load(const FrameLoadRequest&, bool sendNotification, void* notifyData);
        NPError handlePost(const char* url, const char* target, uint32_t len, const char* buf, bool file, void* notifyData, bool sendNotification, bool allowHeaders);
        NPError handlePostReadFile(Vector<char>& buffer, uint32_t len, const char* buf);
        static void freeStringArray(char** stringArray, int length);
        void setCallingPlugin(bool) const;

        void invalidateWindowlessPluginRect(const IntRect&);

        virtual void mediaCanStart();

#if OS(WINDOWS) && ENABLE(NETSCAPE_PLUGIN_API)
        void paintWindowedPluginIntoContext(GraphicsContext*, const IntRect&);
        static HDC WINAPI hookedBeginPaint(HWND, PAINTSTRUCT*);
        static BOOL WINAPI hookedEndPaint(HWND, const PAINTSTRUCT*);
#endif

        static bool platformGetValueStatic(NPNVariable variable, void* value, NPError* result);
        bool platformGetValue(NPNVariable variable, void* value, NPError* result);

        RefPtr<Frame> m_parentFrame;
        RefPtr<PluginPackage> m_plugin;
        Element* m_element;
        bool m_isStarted;
        KURL m_url;
        KURL m_baseURL;
        PluginStatus m_status;
        Vector<IntRect> m_invalidRects;

        void performRequest(PluginRequest*);
        void scheduleRequest(PluginRequest*);
        void requestTimerFired(Timer<PluginView>*);
        void invalidateTimerFired(Timer<PluginView>*);
        Timer<PluginView> m_requestTimer;
        Timer<PluginView> m_invalidateTimer;

        void popPopupsStateTimerFired(Timer<PluginView>*);
        Timer<PluginView> m_popPopupsStateTimer;

#ifdef TPI_FLASH
        // NPN Timers for Flash
        PluginTimers    m_Timers;
#endif
        
        void lifeSupportTimerFired(Timer<PluginView>*);
        Timer<PluginView> m_lifeSupportTimer;

#ifndef NP_NO_CARBON
        bool dispatchNPEvent(NPEvent&);
#endif
        void updatePluginWidget();
        void paintMissingPluginIcon(GraphicsContext*, const IntRect&);

        void handleKeyboardEvent(KeyboardEvent*);
        void handleMouseEvent(MouseEvent*);
#if defined(XP_UNIX) && ENABLE(NETSCAPE_PLUGIN_API)
        void handleFocusInEvent();
        void handleFocusOutEvent();
#endif

#if OS(WINDOWS)
        void paintIntoTransformedContext(HDC);
        PassRefPtr<Image> snapshot();
#endif

        int m_mode;
        int m_paramCount;
        char** m_paramNames;
        char** m_paramValues;
        String m_pluginsPage;

        String m_mimeType;
        WTF::CString m_userAgent;

        NPP m_instance;
        NPP_t m_instanceStruct;
        NPWindow m_npWindow;

        Vector<bool, 4> m_popupStateStack;

        HashSet<RefPtr<PluginStream> > m_streams;
        Vector<PluginRequest*> m_requests;

        bool m_isWindowed;
        bool m_isTransparent;
        bool m_haveInitialized;
        bool m_isWaitingToStart;

#if defined(XP_UNIX)
        bool m_needsXEmbed;
#endif

#if OS(WINDOWS) && ENABLE(NETSCAPE_PLUGIN_API)
        OwnPtr<PluginMessageThrottlerWin> m_messageThrottler;
        WNDPROC m_pluginWndProc;
        unsigned m_lastMessage;
        bool m_isCallingPluginWndProc;
        HDC m_wmPrintHDC;
        bool m_haveUpdatedPluginWidget;
#endif

#if ((PLATFORM(QT) || PLATFORM(WX)) && OS(WINDOWS)) || defined(XP_MACOSX)
        // On Mac OSX and Qt/Windows the plugin does not have its own native widget,
        // but is using the containing window as its reference for positioning/painting.
        PlatformPluginWidget m_window;
public:
        PlatformPluginWidget platformPluginWidget() const { return m_window; }
        void setPlatformPluginWidget(PlatformPluginWidget widget) { m_window = widget; }
#else
public:
        void setPlatformPluginWidget(PlatformPluginWidget widget) { setPlatformWidget(widget); }
        PlatformPluginWidget platformPluginWidget() const { return platformWidget(); }
#endif

private:

#if defined(XP_UNIX) || OS(SYMBIAN)
        void setNPWindowIfNeeded();
#elif defined(XP_MACOSX)
        NP_CGContext m_npCgContext;
        OwnPtr<Timer<PluginView> > m_nullEventTimer;
        NPDrawingModel m_drawingModel;
        NPEventModel m_eventModel;
        CGContextRef m_contextRef;
        WindowRef m_fakeWindow;
#if PLATFORM(QT)
        QPixmap m_pixmap;
#endif

        Point m_lastMousePos;
        void setNPWindowIfNeeded();
        void nullEventTimerFired(Timer<PluginView>*);
        Point globalMousePosForPlugin() const;
        Point mousePosForPlugin(MouseEvent* event = 0) const;
#endif

#if defined(HAS_X11) && HAS_X11 && ENABLE(NETSCAPE_PLUGIN_API)
        bool m_hasPendingGeometryChange;
        Pixmap m_drawable;
        Visual* m_visual;
        Colormap m_colormap;
        Display* m_pluginDisplay;

        void initXEvent(XEvent* event);
#endif

        IntRect m_clipRect; // The clip rect to apply to a windowed plug-in
        IntRect m_windowRect; // Our window rect.

        bool m_loadManually;
        RefPtr<PluginStream> m_manualStream;

        bool m_isJavaScriptPaused;

        bool m_isHalted;
        bool m_hasBeenHalted;

        static PluginView* s_currentPluginView;
    };

} // namespace WebCore

#endif
