/*
 * Copyright (C) 2006 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2006 Michael Emmel mike.emmel@gmail.com
 * All rights reserved.
 * Copyright (C) 2009, 2010, 2011 Espial Group Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef ResourceHandleManager_h
#define ResourceHandleManager_h

#include "Frame.h"
#include "PlatformString.h"
#include "Timer.h"
#include "ResourceHandleClient.h"

#if PLATFORM(WIN)
#include <winsock2.h>
#include <windows.h>
#endif

#include <curl/curl.h>
#include <openssl/x509.h>
#include <wtf/Vector.h>
#include <wtf/text/CString.h>
#include <ResourceHandle.h>
#include "HashMap.h"

namespace WebCore {

class Document;
class KURL;
class Frame;
class ResourceHandle;
class ResourceRequest;

class ResourceHandleManager {
public:
    enum ProxyType {
        HTTP = CURLPROXY_HTTP,
        Socks4 = CURLPROXY_SOCKS4,
        Socks4A = CURLPROXY_SOCKS4A,
        Socks5 = CURLPROXY_SOCKS5,
        Socks5Hostname = CURLPROXY_SOCKS5_HOSTNAME
    };
    static ResourceHandleManager* sharedInstance();
    void add(ResourceHandle*);
    void cancel(ResourceHandle*);

    void setCookieJarFileName(const char* cookieJarFileName);
    void setCookieJarNetTVFileName(const char* cookieJarFileName);
    
    void shutdown();
    
    void resumeAll();
    
    void dispatchSynchronousJob(ResourceHandle*);
    
    void rootCertificatesChanged();

    void dumpCookieJar();
    void setCookie(const char* domain, const char* path, const char* line, bool fixed);
    void deleteCookie(const char* domain, const char* path, const char* name);
    void setProxyInfo(const String& httpHost = "",
                      unsigned long httpPort = 0,
                      const String& httpsHost = "",
                      unsigned long httpsPort = 0,
                      ProxyType type = HTTP);
    
    void setCookies(Document*, const KURL&, const String&);
    String cookies(const Document*, const KURL&);
    
    void downloadTimerCallback(Timer<ResourceHandleManager>*);

    void setCookiesEnabled(bool enable);
    void clearCookies();
    bool cookieEnabled() const { return m_cookiesEnabled; }
    void setExtendedHttpHeaders(const Vector<String>& headers, const String& domain);

    void initializeHttpAuthentication(ResourceHandle*);
    void initializeProxy(ResourceHandle*) const;
    void initializeExtendedHeaders(ResourceHandle*);
        
private:
    ResourceHandleManager();
    ~ResourceHandleManager();
    void removeFromCurl(ResourceHandle*);
    bool removeScheduledJob(ResourceHandle*);
    void startJob(ResourceHandle*);
    bool startScheduledJobs();

    void initializeHandle(ResourceHandle*);
    
    static int ssl_verify_callback(int preverify_ok, X509_STORE_CTX *ctx);
    
    static ResourceHandle* findHandle(CURL* curl /* easy handle */);
    
    static CURLcode ssl_ctx_callback(	CURL *curl,    /* easy handle */
    									void *ssl_ctx, /* actually an OpenSSL SSL_CTX */
								        void *ssl,     /* actually an OpenSSL SSL */
								        void *userptr);
    
    static size_t headerCallback(char* ptr, size_t size, size_t nmemb, void* data);
    
    static void curl_ready_http_callback(CURL *curl, /* easy handler */
    										void* SSL /*if secure*/);
    
    static int curl_send_cookie_function(CURL *curl,       /* easy handler */
								  		const char* domain,
								        const char* name,
								        int cookie_type);
    
    Timer<ResourceHandleManager> m_downloadTimer;
    CURLM* m_curlMultiHandle;
    CURLSH* m_curlShareHandle;
    bool m_rootCertificatesChanged;
    ResourceHandle* m_syncResourceHandle;
    char* m_cookieJarFileName;
    char* m_cookieJarNetTVFileName;
    char m_curlErrorBuffer[CURL_ERROR_SIZE];
    Vector<ResourceHandle*> m_scheduledResourceHandles;
    Vector<ResourceHandle*> m_runningResourceHandles;
    const CString m_certificatePath;

    bool m_cookiesEnabled;
    String m_httpProxy;
    String m_httpsProxy;

    HashMap<String, Vector<String> > m_extenedHttpHeaders;
    Vector<String> m_exceptionalExtenedHttpHeaders;

    ProxyType m_proxyType;
};

}

#endif
