/*
 * Copyright (C) 2006 Apple Computer, Inc.  All rights reserved.
 * Copyright (C) 2009, 2010, 2011 Espial Group Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE COMPUTER, INC. ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL APPLE COMPUTER, INC. OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY
 * OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */

#ifndef PNGImageDecoder_h
#define PNGImageDecoder_h

#include "ImageDecoder.h"
#include <wtf/OwnPtr.h>
#include "png.h"

namespace WebCore {


class PNGImageReader
{
public:
    PNGImageReader(PNGImageDecoder* decoder);
    virtual ~PNGImageReader();

    unsigned currentBufferSize() const { return m_currentBufferSize; }

    virtual bool decode(const SharedBuffer& data, bool sizeOnly);

    bool decodingSizeOnly() const { return m_decodingSizeOnly; }
    png_structp pngPtr() const { return m_png; }
    png_infop infoPtr() const { return m_info; }
    png_bytep interlaceBuffer() const { return m_interlaceBuffer; }
    bool hasAlpha() const { return m_hasAlpha; }

    void setReadOffset(unsigned offset) { m_readOffset = offset; }
    void setHasAlpha(bool b) { m_hasAlpha = b; }

    void createInterlaceBuffer(int size) { tryFastMalloc(size).getValue(m_interlaceBuffer); }

protected:
    unsigned m_readOffset;
    bool m_decodingSizeOnly;
    png_structp m_png;
    png_infop m_info;
    png_bytep m_interlaceBuffer;
    bool m_hasAlpha;
    unsigned m_currentBufferSize;
};

// This class decodes the PNG image format.
class PNGImageDecoder : public ImageDecoder {
public:
    PNGImageDecoder();
    virtual ~PNGImageDecoder();

    // ImageDecoder
    virtual String filenameExtension() const { return "png"; }
    virtual bool isSizeAvailable();
    virtual bool setSize(unsigned width, unsigned height);
    virtual RGBA32Buffer* frameBufferAtIndex(size_t index);

    // Callbacks from libpng
    void headerAvailable();
    void rowAvailable(unsigned char* rowBuffer, unsigned rowIndex, int interlacePass);
    void pngComplete();

    bool isComplete() const
    {
        return !m_frameBufferCache.isEmpty() && (m_frameBufferCache.first().status() == RGBA32Buffer::FrameComplete);
    }

protected:
    // Decodes the image.  If |onlySize| is true, stops decoding after
    // calculating the image size.  If decoding fails but there is no more
    // data coming, sets the "decode failure" flag.
    virtual void decode(bool onlySize);

protected:
    OwnPtr<PNGImageReader> m_reader;
};

} // namespace WebCore

#endif
