/*
 * This file is part of the internal font implementation.  It should not be included by anyone other than
 * FontMac.cpp, FontWin.cpp and Font.cpp.
 *
 * Copyright (C) 2006 Apple Computer, Inc.
 * Copyright (C) 2006 Michael Emmel mike.emmel@gmail.com
 * Copyright (C) 2007 Holger Hans Peter Freyther
 * Copyright (C) 2007 Pioneer Research Center USA, Inc.
 * All rights reserved.
 * Copyright (C) 2009, 2010, 2011 Espial Group Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 */

#ifndef FontPlatformData_h
#define FontPlatformData_h

#include "GlyphBuffer.h"
#include "FontDescription.h"
#include <cairo.h>
#include <cairo-ft.h>
#include <fontconfig/fcfreetype.h>

namespace WebCore {

class PrivateFont;
class GraphicsContext;
class GlyphBuffer;
class GlyphPage;
class FloatPoint;
class TextRun;

struct FontMetrics
{
    int ascent;
    int descent;
    int lineSpacing;
    float xHeight;
    float spaceWidth;
};

class FontPlatformData {
public:
    FontPlatformData(float size=0, bool bold=false, bool italic=false);
    FontPlatformData(cairo_font_face_t* fontFace, int size, bool bold, bool italic);
    FontPlatformData(const FontDescription&, FcPattern*);

    FontPlatformData(const FontDescription&, PassRefPtr<PrivateFont>);
    
    FontPlatformData(const FontPlatformData&);
    FontPlatformData& operator=(const FontPlatformData&);
    
    FontPlatformData smallCapsFontData(const FontDescription& fontDescription) const;
    
    FontMetrics metrics() const;
        
    bool containsCharacters(const UChar* characters, int length) const;
    
    float platformWidthForGlyph(Glyph glyph) const;
    
    bool fill(GlyphPage&, unsigned offset, unsigned length, UChar*, unsigned bufferLength, const SimpleFontData*) const;
    
    void drawGlyphs(GraphicsContext*, const GlyphBuffer&, const TextRun&, int from, int numGlyphs, const FloatPoint&) const;
    		
    ~FontPlatformData();
    
    void invalidate() const;

    static bool init();

    bool isFixedPitch();
    float size() const { return m_size; }
    
    bool syntheticBold() const { return m_syntheticBold; }
    bool syntheticOblique() const { return m_syntheticOblique; }
    
    float syntheticBoldOffset() const { return syntheticBold() ? 1.0f : 0.f; }
    
    bool operator==(const FontPlatformData&) const;
    
#ifndef NDEBUG
    String description() const;
#endif
    
private:
    FcPattern* m_pattern;
    cairo_scaled_font_t* m_scaledFont;
    
    mutable RefPtr<PrivateFont> m_privateFont;
    
    float m_size;
    bool m_syntheticBold;
    bool m_syntheticOblique;
    mutable float m_spaceWidth;
};

}

#endif
