/*
 * Copyright (C) 2008 Apple Inc. All rights reserved.
 * Copyright (C) 2010 Espial Group Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 * 3.  Neither the name of Apple Computer, Inc. ("Apple") nor the names of
 *     its contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE AND ITS CONTRIBUTORS "AS IS" AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE OR ITS CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "RegisterFile.h"
#include <wtf/MemoryMonitor.h>
#include "JSGlobalObject.h"

namespace JSC {

RegisterFile::RegisterFile(size_t capacity, size_t maxGlobals)
: m_numGlobals(0)
, m_maxGlobals(maxGlobals)
, m_start(0)
, m_end(0)
, m_max(0)
, m_buffer(0)
{
    // Verify that our values will play nice with mmap and VirtualAlloc.
    ASSERT(isPageAligned(maxGlobals));
    ASSERT(isPageAligned(capacity));

    size_t bufferLength = (capacity + maxGlobals) * sizeof(Register);
#if HAVE(MMAP)
    m_buffer = static_cast<Register*>(mmap(0, bufferLength, PROT_READ|PROT_WRITE, MAP_PRIVATE|MAP_ANON, VM_TAG_FOR_REGISTERFILE_MEMORY, 0));
    
    WTF::JSMmapMonitor::instance()->mmapStat(m_buffer, bufferLength); 
    
    if (m_buffer == MAP_FAILED) {
#if OS(WINCE)
        fprintf(stderr, "Could not allocate register file: %d\n", GetLastError());
#else
        fprintf(stderr, "Could not allocate register file: %d\n", errno);
#endif
        CRASH();
    }
#elif HAVE(VIRTUALALLOC)
    m_buffer = static_cast<Register*>(VirtualAlloc(0, roundUpAllocationSize(bufferLength, commitSize), MEM_RESERVE, PAGE_READWRITE));
    if (!m_buffer) {
#if OS(WINCE)
        fprintf(stderr, "Could not allocate register file: %d\n", GetLastError());
#else
        fprintf(stderr, "Could not allocate register file: %d\n", errno);
#endif
        CRASH();
    }
    size_t committedSize = roundUpAllocationSize(maxGlobals * sizeof(Register), commitSize);
    void* commitCheck = VirtualAlloc(m_buffer, committedSize, MEM_COMMIT, PAGE_READWRITE);
    if (commitCheck != m_buffer) {
#if OS(WINCE)
        fprintf(stderr, "Could not allocate register file: %d\n", GetLastError());
#else
        fprintf(stderr, "Could not allocate register file: %d\n", errno);
#endif
        CRASH();
    }
    m_commitEnd = reinterpret_cast<Register*>(reinterpret_cast<char*>(m_buffer) + committedSize);
#else 
    /* 
     * If neither MMAP nor VIRTUALALLOC are available - use fastMalloc instead.
     *
     * Please note that this is the fallback case, which is non-optimal.
     * If any possible, the platform should provide for a better memory
     * allocation mechanism that allows for "lazy commit" or dynamic
     * pre-allocation, similar to mmap or VirtualAlloc, to avoid waste of memory.
     */
    m_buffer = static_cast<Register*>(fastMalloc(bufferLength));
#endif
    m_start = m_buffer + maxGlobals;
    m_end = m_start;
    m_maxUsed = m_end;
    m_max = m_start + capacity;
}

RegisterFile::~RegisterFile()
{
#if HAVE(MMAP)
    size_t size = ((m_max - m_start) + m_maxGlobals) * sizeof(Register);
    
    if(munmap(m_buffer, size) == 0){
    	WTF::JSMmapMonitor::instance()->munmapStat(m_buffer, size);
    }
#elif HAVE(VIRTUALALLOC)
#if OS(WINCE)
    VirtualFree(m_buffer, DWORD(m_commitEnd) - DWORD(m_buffer), MEM_DECOMMIT);
#endif
    VirtualFree(m_buffer, 0, MEM_RELEASE);
#else
    fastFree(m_buffer);
#endif
}

void RegisterFile::releaseExcessCapacity()
{
#if HAVE(MMAP) && HAVE(MADV_FREE) && !HAVE(VIRTUALALLOC)
    while (madvise(m_start, (m_max - m_start) * sizeof(Register), MADV_FREE) == -1 && errno == EAGAIN) { }
#elif HAVE(VIRTUALALLOC)
    VirtualFree(m_start, (m_max - m_start) * sizeof(Register), MEM_DECOMMIT);
    m_commitEnd = m_start;
#endif
    m_maxUsed = m_start;
}

void RegisterFile::setGlobalObject(JSGlobalObject* globalObject)
{
    m_globalObject = globalObject;
}

bool RegisterFile::clearGlobalObject(JSGlobalObject* globalObject)
{
    return m_globalObject.clear(globalObject);
}

JSGlobalObject* RegisterFile::globalObject()
{
    return m_globalObject.get();
}

} // namespace JSC
